/*:
 * @target MZ
 * @plugindesc v1.1 フルスクリーン時のボケ対策：内部解像度を画面に合わせて自動調整（Smooth / Pixel切替）
 * @author HS
 *
 * @param Mode
 * @text 表示モード
 * @type select
 * @option Smooth（文字なめらか・にじみ抑制）
 * @value smooth
 * @option Pixel（補間OFFで超クッキリ）
 * @value pixel
 * @default smooth
 *
 * @param MaxResolutionScale
 * @text 最大内部解像度倍率
 * @type number
 * @min 1
 * @max 4
 * @default 3
 *
 * @help
 * ■概要
 * ・フルスクリーン/リサイズに応じて内部解像度(resolution)とCSSサイズを再計算し、ボケを抑えます。
 * ・アスペクト比維持（黒帯）。入力座標も崩しません。
 * ・Mode=smooth：LINEAR補間＋高解像度描画
 * ・Mode=pixel ：NEAREST補間＋CSS image-rendering: pixelated
 *
 * ■並び順
 * 画面/キャンバスを触るプラグインの「一番下」推奨。
 */
(() => {
  "use strict";
  const PN = "HS_FullscreenScaler";
  const PRM = PluginManager.parameters(PN);
  const MODE = String(PRM["Mode"] || "smooth");
  const MAXS = Math.max(1, Math.min(4, Number(PRM["MaxResolutionScale"] || 3)));

  function isFullscreen() {
    return !!(document.fullscreenElement || document.webkitFullscreenElement);
  }

  function applyScale() {
    const app = Graphics && Graphics._app;
    if (!app) return;
    const view = app.view;

    const baseW = Graphics.width;
    const baseH = Graphics.height;

    const vw = window.innerWidth;
    const vh = window.innerHeight;
    const scale = Math.min(vw / baseW, vh / baseH);

    // 内部解像度倍率：フルスクリーン中は見た目のscaleに寄せる
    let res = Math.max(1, Math.min(MAXS, scale * (window.devicePixelRatio || 1)));
    if (isFullscreen()) res = Math.max(1, Math.min(MAXS, scale));

    // 補間モード
    if (window.PIXI) {
      PIXI.settings.SCALE_MODE = (MODE === "pixel")
        ? PIXI.SCALE_MODES.NEAREST
        : PIXI.SCALE_MODES.LINEAR;
      if (app.renderer) app.renderer.roundPixels = (MODE === "pixel");
    }

    // 内部解像度更新
    if (app.renderer && app.renderer.resolution !== res) {
      app.renderer.resolution = res;
      app.renderer.resize(baseW, baseH);
    }

    // CSSサイズと中央寄せ（黒帯）
    const cssW = Math.floor(baseW * scale);
    const cssH = Math.floor(baseH * scale);
    const left = Math.floor((vw - cssW) / 2);
    const top  = Math.floor((vh - cssH) / 2);

    const s = view.style;
    s.position = "absolute";
    s.width  = cssW + "px";
    s.height = cssH + "px";
    s.left   = left + "px";
    s.top    = top + "px";
    s.imageRendering = (MODE === "pixel") ? "pixelated" : "auto";

    // 入力座標補正
    Graphics._realScale = scale;
  }

  // **安全フック**：Scene_Boot開始後に存在チェックしてフック
  function installResizeHooksSafely() {
    // Graphics._onResize が関数ならラップ、無ければ window.resize を使う
    if (typeof Graphics._onResize === "function") {
      const orig = Graphics._onResize;
      Graphics._onResize = function() {
        orig.apply(this, arguments);
        setTimeout(applyScale, 0);
      };
    } else {
      window.addEventListener("resize", () => setTimeout(applyScale, 0));
    }
    // フルスクリーン変化
    document.addEventListener("fullscreenchange", () => setTimeout(applyScale, 0));
    document.addEventListener("webkitfullscreenchange", () => setTimeout(applyScale, 0));
  }

  // 起動時：必ずScene_Bootの後にフック＆初回適用
  const _bootStart = Scene_Boot.prototype.start;
  Scene_Boot.prototype.start = function() {
    _bootStart.call(this);
    installResizeHooksSafely();
    setTimeout(applyScale, 0);
  };

  // たまに他プラグインがstyleを上書きする対策で、定期的に再適用
  const _updateScene = SceneManager.updateScene;
  SceneManager.updateScene = function() {
    _updateScene.call(this);
    this._hsfsTicker = (this._hsfsTicker || 0) + 1;
    if ((this._hsfsTicker % 120) === 0) applyScale();
  };
})();
